function [ DropLineX, DropLineY , DeviceLineX, DeviceLineY ] = Graphics_PlotSuspendedDrops_V03( Rm, height, thickness, display )
%Function_Pendant_Drop_PressureVolume_V01
% This functions calculates the outline of a pendant drop 
% The function returns both an outline of the drop and
% an outline of the plastic part. By default the plot is placed
% in the bottom right secctor of the graph space, i.e. max Y is 0
% and min X is 0
% INPUTS:
% - Rm      radius 
% - height of the drop
% - display turns the display on or off

global rho g gamma ;

% Make some of the simulation parameters
K             = Rm * (rho * g / gamma)^0.5 ; % shape factor
deltapmax     = height / Rm                ; % Maximum drop size
ns            = 1000                       ; % number of steps in arc length
zsign         = 1                          ; % 1 for pendant drop, -1 for pendant bubble

% Make the outputs
DropLineX     = zeros(2*ns,1) ;
DeviceLineX   = zeros(2*ns,1) ;
DropLineY     = zeros(2*ns,1) ;
DeviceLineY   = zeros(2*ns,1) ;

% the next two parameters can be adjusted to integral numbers larger than 1
% to make sure that sensible initial guesses are always obtained
ndeltapoints = 1 ;  % number of steps used to reach target delta

% Get the drop size
deldelta       = deltapmax/ndeltapoints ; % This is the step size

% parmpsol(1) is the overall countour length, normalized by Rm
% parmpsol(2) is the normalized pressure
% our initial guess, used only for the lowest value of delta, is a
% spherical cap
deltaiter    = 1                   ;
deltap       = deldelta            ;
s            = linspace(1/ns,1,ns) ; % this sets the original mesh of points equally spaced along the undeformed arc length

% in the following initial guess we assume that the membrane is deformed to a constant
% radius of curvature, which is determined by deltap
amax          = 2*atan(deltap)  ; % this is the angle at s=1 (R=Rm)
ap            = s*amax          ; 
rp            = s.*sinc(ap/pi)  ;
zp            = tan(ap/2)       ;
solp(1,:)     = ap              ; % these are the angles along the whole membrane profile for the assumed cirular cross section
solp(2,:)     = rp              ; % these are the r values
solp(3,:)     = zp              ; % these are the z values
parmpsol(1)   = 1/sinc(amax/pi) ; % initial guess for lp (normalized arc length of membrane)
parmpsol(2)   = 4*deltap/(1+deltap^2);  % inital guess for normalized pressure

% in the following loop we solve the pendant drop problem for a constant
% interfacial tension
while deltaiter<=ndeltapoints;
    parmpguess      = parmpsol;
    solpguess       = solp;
    [parmpsol,solp] = Function_Pendantliquid_v03(ns,deltap,K,zsign,parmpguess,solpguess);         
    deltaiter       = deltaiter+1;
    deltap          = deltap+deldelta;
end

% the solution to the undeformed case is alrady known:
rp(:)=solp(2,:);
zp(:)=solp(3,:);

% Make a figure
if (display == true)
    figure(10)                 ;
    plot(rp,zp-zp(ns),'--b')   ;
    hold on
    plot(-rp,zp-zp(ns),'--b')  ;
end

% Make the device Outline
if (thickness > 0)
    
    % Make a Drop Outline
    DropLineX(1:ns)       = Rm*(rp+1.5)                 ;
    DropLineX(ns+1:2*ns)  = Rm*(-rp+1.5)                ;    
    DropLineY(1:ns)       = Rm*(zp-zp(ns)) - thickness  ;
    DropLineY(ns+1:2*ns)  = Rm*(zp-zp(ns)) - thickness  ;

    % Make the device Outline
    DeviceLineX(1:ns/2)        = Rm*linspace(0,0.5,ns/2)      ;
    DeviceLineX(ns/2+1:ns)     = Rm * 0.5                     ;
    DeviceLineX(ns+1:3*ns/2)   = Rm*linspace(2.5,3,ns/2)      ;
    DeviceLineX(3*ns/2+1:2*ns) = Rm * 2.5                     ; 
    DeviceLineY(1:ns/2)        = -thickness                    ;
    DeviceLineY(ns/2+1:ns)     = -thickness*linspace(0,1,ns/2) ;
    DeviceLineY(ns+1:3*ns/2)   = -thickness                    ;
    DeviceLineY(3*ns/2+1:2*ns) = -thickness*linspace(0,1,ns/2) ;
    
else
    
    % Make a Drop Outline
    DropLineX(1:ns)       = Rm*(rp+1.5)                 ;
    DropLineX(ns+1:2*ns)  = Rm*(-rp+1.5)                ;    
    DropLineY(1:ns)       = Rm*(zp-zp(ns)) - thickness  ;
    DropLineY(ns+1:2*ns)  = Rm*(zp-zp(ns)) - thickness  ;

    % Make the device Outline
    DeviceLineX(1:ns/2)        = Rm*linspace(0,0.5,ns/2)      ;
    DeviceLineX(ns/2+1:ns)     = Rm * 0.5                     ;
    DeviceLineX(ns+1:3*ns/2)   = Rm*linspace(2.5,3,ns/2)      ;
    DeviceLineX(3*ns/2+1:2*ns) = Rm * 2.5                     ; 
    DeviceLineY(1:ns/2)        = -thickness                    ;
    DeviceLineY(ns/2+1:ns)     = -thickness*linspace(0,1,ns/2) ;
    DeviceLineY(ns+1:3*ns/2)   = -thickness                    ;
    DeviceLineY(3*ns/2+1:2*ns) = -thickness*linspace(0,1,ns/2) ;
    
end

end

